﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/workdocs/WorkDocsRequest.h>
#include <aws/workdocs/WorkDocs_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace WorkDocs {
namespace Model {

/**
 */
class DescribeGroupsRequest : public WorkDocsRequest {
 public:
  AWS_WORKDOCS_API DescribeGroupsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeGroups"; }

  AWS_WORKDOCS_API Aws::String SerializePayload() const override;

  AWS_WORKDOCS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_WORKDOCS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Amazon WorkDocs authentication token. Not required when using Amazon Web
   * Services administrator credentials to access the API.</p>
   */
  inline const Aws::String& GetAuthenticationToken() const { return m_authenticationToken; }
  inline bool AuthenticationTokenHasBeenSet() const { return m_authenticationTokenHasBeenSet; }
  template <typename AuthenticationTokenT = Aws::String>
  void SetAuthenticationToken(AuthenticationTokenT&& value) {
    m_authenticationTokenHasBeenSet = true;
    m_authenticationToken = std::forward<AuthenticationTokenT>(value);
  }
  template <typename AuthenticationTokenT = Aws::String>
  DescribeGroupsRequest& WithAuthenticationToken(AuthenticationTokenT&& value) {
    SetAuthenticationToken(std::forward<AuthenticationTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A query to describe groups by group name.</p>
   */
  inline const Aws::String& GetSearchQuery() const { return m_searchQuery; }
  inline bool SearchQueryHasBeenSet() const { return m_searchQueryHasBeenSet; }
  template <typename SearchQueryT = Aws::String>
  void SetSearchQuery(SearchQueryT&& value) {
    m_searchQueryHasBeenSet = true;
    m_searchQuery = std::forward<SearchQueryT>(value);
  }
  template <typename SearchQueryT = Aws::String>
  DescribeGroupsRequest& WithSearchQuery(SearchQueryT&& value) {
    SetSearchQuery(std::forward<SearchQueryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the organization.</p>
   */
  inline const Aws::String& GetOrganizationId() const { return m_organizationId; }
  inline bool OrganizationIdHasBeenSet() const { return m_organizationIdHasBeenSet; }
  template <typename OrganizationIdT = Aws::String>
  void SetOrganizationId(OrganizationIdT&& value) {
    m_organizationIdHasBeenSet = true;
    m_organizationId = std::forward<OrganizationIdT>(value);
  }
  template <typename OrganizationIdT = Aws::String>
  DescribeGroupsRequest& WithOrganizationId(OrganizationIdT&& value) {
    SetOrganizationId(std::forward<OrganizationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The marker for the next set of results. (You received this marker from a
   * previous call.)</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeGroupsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to return with this call.</p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline DescribeGroupsRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_authenticationToken;
  bool m_authenticationTokenHasBeenSet = false;

  Aws::String m_searchQuery;
  bool m_searchQueryHasBeenSet = false;

  Aws::String m_organizationId;
  bool m_organizationIdHasBeenSet = false;

  Aws::String m_marker;
  bool m_markerHasBeenSet = false;

  int m_limit{0};
  bool m_limitHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkDocs
}  // namespace Aws
