/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.opinion;

import ghidra.app.util.MemoryBlockUtils;
import ghidra.app.util.bin.ByteProvider;
import ghidra.app.util.bin.format.unixaout.UnixAoutHeader;
import ghidra.app.util.bin.format.unixaout.UnixAoutRelocation;
import ghidra.app.util.bin.format.unixaout.UnixAoutRelocationTable;
import ghidra.app.util.bin.format.unixaout.UnixAoutStringTable;
import ghidra.app.util.bin.format.unixaout.UnixAoutSymbol;
import ghidra.app.util.bin.format.unixaout.UnixAoutSymbolTable;
import ghidra.app.util.importer.MessageLog;
import ghidra.program.database.function.OverlappingFunctionException;
import ghidra.program.database.mem.FileBytes;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.listing.FunctionManager;
import ghidra.program.model.listing.Program;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryAccessException;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.reloc.Relocation;
import ghidra.program.model.reloc.RelocationTable;
import ghidra.program.model.symbol.SourceType;
import ghidra.program.model.symbol.Symbol;
import ghidra.program.model.symbol.SymbolIterator;
import ghidra.program.model.symbol.SymbolTable;
import ghidra.program.model.util.CodeUnitInsertionException;
import ghidra.util.DataConverter;
import ghidra.util.MonitoredInputStream;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.exception.InvalidInputException;
import ghidra.util.task.TaskMonitor;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

public class UnixAoutProgramLoader {
    private final int EXTERNAL_BLOCK_MIN_SIZE = 65536;
    public static final String dot_text = ".text";
    public static final String dot_data = ".data";
    public static final String dot_bss = ".bss";
    public static final String dot_rel_text = ".rel.text";
    public static final String dot_rel_data = ".rel.data";
    public static final String dot_strtab = ".strtab";
    public static final String dot_symtab = ".symtab";
    private final Program program;
    private final TaskMonitor monitor;
    private final MessageLog log;
    private final UnixAoutHeader header;
    private FileBytes fileBytes;
    private UnixAoutRelocationTable relText;
    private UnixAoutRelocationTable relData;
    private UnixAoutSymbolTable symtab;
    private UnixAoutStringTable strtab;
    private Map<String, Long> possibleBssSymbols = new HashMap<String, Long>();
    private int extraBssSize = 0;
    private int undefinedSymbolCount = 0;

    public UnixAoutProgramLoader(Program program, UnixAoutHeader header, TaskMonitor monitor, MessageLog log) {
        this.program = program;
        this.monitor = monitor;
        this.log = log;
        this.header = header;
    }

    public void loadAout(long baseAddr) throws IOException, CancelledException {
        this.log.appendMsg(String.format("----- Loading %s -----", this.header.getReader().getByteProvider().getAbsolutePath()));
        this.log.appendMsg(String.format("Found a.out type %s.", this.header.getExecutableType().name()));
        ByteProvider byteProvider = this.header.getReader().getByteProvider();
        try {
            this.buildTables(byteProvider);
            this.preprocessSymbolTable();
            this.loadSections(baseAddr, byteProvider);
            this.loadSymbols();
            this.applyRelocations(baseAddr, this.program.getMemory().getBlock(dot_text), this.relText);
            this.applyRelocations(baseAddr, this.program.getMemory().getBlock(dot_data), this.relData);
            this.markupSections();
        }
        catch (AddressOverflowException | MemoryAccessException | CodeUnitInsertionException | DuplicateNameException | InvalidInputException e) {
            throw new RuntimeException(e);
        }
    }

    private void buildTables(ByteProvider byteProvider) throws IOException {
        if (this.header.getStrSize() > 0L) {
            this.strtab = new UnixAoutStringTable(this.header.getReader(), this.header.getStrOffset(), this.header.getStrSize());
        }
        if (this.header.getSymSize() > 0L) {
            this.symtab = new UnixAoutSymbolTable(this.header.getReader(), this.header.getSymOffset(), this.header.getSymSize(), this.strtab, this.log);
        }
        if (this.header.getTextRelocSize() > 0L) {
            this.relText = new UnixAoutRelocationTable(this.header.getReader(), this.header.getTextRelocOffset(), this.header.getTextRelocSize(), this.symtab);
        }
        if (this.header.getDataRelocSize() > 0L) {
            this.relData = new UnixAoutRelocationTable(this.header.getReader(), this.header.getDataRelocOffset(), this.header.getDataRelocSize(), this.symtab);
        }
    }

    private void preprocessSymbolTable() {
        if (this.symtab == null) {
            return;
        }
        boolean foundStabs = false;
        for (UnixAoutSymbol symbol : this.symtab) {
            switch (symbol.type) {
                case N_UNDF: {
                    if (symbol.value > 0L) {
                        this.possibleBssSymbols.put(symbol.name, symbol.value);
                        break;
                    }
                    ++this.undefinedSymbolCount;
                    break;
                }
                case N_STAB: {
                    if (foundStabs) break;
                    foundStabs = true;
                    this.log.appendMsg(dot_symtab, "File contains STABS.");
                    break;
                }
            }
        }
        for (Long value : this.possibleBssSymbols.values()) {
            this.extraBssSize = (int)((long)this.extraBssSize + value);
        }
        if (this.extraBssSize > 0) {
            this.log.appendMsg(dot_bss, String.format("Added %d bytes for N_UNDF symbols.", this.extraBssSize));
        }
    }

    private void loadSections(long baseAddr, ByteProvider byteProvider) throws AddressOverflowException, IOException, CancelledException {
        Address address;
        this.monitor.setMessage("Loading FileBytes...");
        try (InputStream fileIn = byteProvider.getInputStream(0L);
             MonitoredInputStream mis = new MonitoredInputStream(fileIn, this.monitor);){
            mis.setCleanupOnCancel(false);
            this.fileBytes = this.program.getMemory().createFileBytes(byteProvider.getName(), 0L, byteProvider.length(), (InputStream)mis, this.monitor);
        }
        AddressSpace defaultAddressSpace = this.program.getAddressFactory().getDefaultAddressSpace();
        Address otherAddress = AddressSpace.OTHER_SPACE.getMinAddress();
        Address nextFreeAddress = defaultAddressSpace.getAddress(0L);
        if (this.header.getTextOffset() != 0L || this.header.getTextSize() < 32L) {
            MemoryBlockUtils.createInitializedBlock(this.program, true, "_aoutHeader", otherAddress, this.fileBytes, 0L, 32L, null, null, false, false, false, this.log);
        }
        if (this.header.getTextSize() > 0L) {
            address = defaultAddressSpace.getAddress(baseAddr + this.header.getTextAddr());
            nextFreeAddress = address.add(this.header.getTextSize());
            MemoryBlockUtils.createInitializedBlock(this.program, false, dot_text, address, this.fileBytes, this.header.getTextOffset(), this.header.getTextSize(), null, null, true, true, true, this.log);
        }
        if (this.header.getDataSize() > 0L) {
            address = defaultAddressSpace.getAddress(baseAddr + this.header.getDataAddr());
            nextFreeAddress = address.add(this.header.getDataSize());
            MemoryBlockUtils.createInitializedBlock(this.program, false, dot_data, address, this.fileBytes, this.header.getDataOffset(), this.header.getDataSize(), null, null, true, true, false, this.log);
        }
        if (this.header.getBssSize() + (long)this.extraBssSize > 0L) {
            address = defaultAddressSpace.getAddress(baseAddr + this.header.getBssAddr());
            nextFreeAddress = address.add(this.header.getBssSize() + (long)this.extraBssSize);
            MemoryBlockUtils.createUninitializedBlock(this.program, false, dot_bss, address, this.header.getBssSize() + (long)this.extraBssSize, null, null, true, true, false, this.log);
        }
        if (this.undefinedSymbolCount > 0) {
            MemoryBlock externalBlock;
            int externalSectionSize = this.undefinedSymbolCount * 4;
            if (externalSectionSize < 65536) {
                externalSectionSize = 65536;
            }
            if ((externalBlock = MemoryBlockUtils.createUninitializedBlock(this.program, false, "EXTERNAL", nextFreeAddress, externalSectionSize, null, null, false, false, false, this.log)) != null) {
                externalBlock.setArtificial(true);
            }
        }
        if (this.header.getStrSize() > 0L) {
            MemoryBlockUtils.createInitializedBlock(this.program, true, dot_strtab, otherAddress, this.fileBytes, this.header.getStrOffset(), this.header.getStrSize(), null, null, false, false, false, this.log);
        }
        if (this.header.getSymSize() > 0L) {
            MemoryBlockUtils.createInitializedBlock(this.program, true, dot_symtab, otherAddress, this.fileBytes, this.header.getSymOffset(), this.header.getSymSize(), null, null, false, false, false, this.log);
        }
        if (this.header.getTextRelocSize() > 0L) {
            MemoryBlockUtils.createInitializedBlock(this.program, true, dot_rel_text, otherAddress, this.fileBytes, this.header.getTextRelocOffset(), this.header.getTextRelocSize(), null, null, false, false, false, this.log);
        }
        if (this.header.getDataRelocSize() > 0L) {
            MemoryBlockUtils.createInitializedBlock(this.program, true, dot_rel_data, otherAddress, this.fileBytes, this.header.getDataRelocOffset(), this.header.getDataRelocSize(), null, null, false, false, false, this.log);
        }
    }

    private void loadSymbols() throws InvalidInputException {
        this.monitor.setMessage("Loading symbols...");
        if (this.symtab == null) {
            return;
        }
        SymbolTable symbolTable = this.program.getSymbolTable();
        FunctionManager functionManager = this.program.getFunctionManager();
        MemoryBlock textBlock = this.program.getMemory().getBlock(dot_text);
        MemoryBlock dataBlock = this.program.getMemory().getBlock(dot_data);
        MemoryBlock bssBlock = this.program.getMemory().getBlock(dot_bss);
        MemoryBlock externalBlock = this.program.getMemory().getBlock("EXTERNAL");
        int extraBssOffset = 0;
        int undefinedSymbolIdx = 0;
        ArrayList<String> aliases = new ArrayList<String>();
        for (UnixAoutSymbol symbol : this.symtab) {
            Address address = null;
            MemoryBlock block = null;
            switch (symbol.type) {
                case N_TEXT: {
                    address = textBlock != null ? textBlock.getStart().add(symbol.value) : null;
                    block = textBlock;
                    break;
                }
                case N_DATA: {
                    address = dataBlock != null ? dataBlock.getStart().add(symbol.value) : null;
                    block = dataBlock;
                    break;
                }
                case N_BSS: {
                    address = bssBlock != null ? bssBlock.getStart().add(symbol.value) : null;
                    block = bssBlock;
                    break;
                }
                case N_UNDF: {
                    if (symbol.value > 0L) {
                        address = bssBlock.getEnd().add((long)extraBssOffset);
                        block = bssBlock;
                        extraBssOffset = (int)((long)extraBssOffset + symbol.value);
                        break;
                    }
                    address = externalBlock.getStart().add((long)(undefinedSymbolIdx++ * 4));
                    block = externalBlock;
                    symbolTable.addExternalEntryPoint(address);
                    break;
                }
                case N_INDR: {
                    aliases.add(symbol.name);
                    break;
                }
                case N_STAB: 
                case N_ABS: 
                case N_FN: 
                case UNKNOWN: {
                    aliases.clear();
                }
            }
            if (address == null || block == null) continue;
            switch (symbol.kind) {
                case AUX_FUNC: {
                    try {
                        functionManager.createFunction(symbol.name, address, (AddressSetView)new AddressSet(address), SourceType.IMPORTED);
                    }
                    catch (OverlappingFunctionException e) {
                        this.log.appendMsg(block.getName(), String.format("Failed to create function %s @ %s, creating symbol instead.", symbol.name, address));
                        symbolTable.createLabel(address, symbol.name, SourceType.IMPORTED);
                    }
                    break;
                }
                default: {
                    Symbol label = symbolTable.createLabel(address, symbol.name, SourceType.IMPORTED);
                    if (!symbol.isExt) break;
                    label.setPrimary();
                }
            }
            for (String alias : aliases) {
                symbolTable.createLabel(address, alias, SourceType.IMPORTED);
            }
            aliases.clear();
        }
    }

    private void applyRelocations(long baseAddr, MemoryBlock targetBlock, UnixAoutRelocationTable relTable) throws MemoryAccessException {
        if (targetBlock == null || relTable == null) {
            return;
        }
        this.monitor.setMessage(String.format("Applying relocations for section %s...", targetBlock.getName()));
        DataConverter dc = DataConverter.getInstance((boolean)this.program.getLanguage().isBigEndian());
        SymbolTable symbolTable = this.program.getSymbolTable();
        RelocationTable relocationTable = this.program.getRelocationTable();
        Memory memory = this.program.getMemory();
        MemoryBlock textBlock = memory.getBlock(dot_text);
        MemoryBlock dataBlock = memory.getBlock(dot_data);
        MemoryBlock bssBlock = memory.getBlock(dot_bss);
        int idx = 0;
        for (UnixAoutRelocation relocation : relTable) {
            Address targetAddress = targetBlock.getStart().add(relocation.address);
            byte[] originalBytes = new byte[relocation.pointerLength];
            targetBlock.getBytes(targetAddress, originalBytes);
            long addend = dc.getValue(originalBytes, 0, (int)relocation.pointerLength);
            Long value = null;
            Relocation.Status status = Relocation.Status.FAILURE;
            if (relocation.baseRelative || relocation.jmpTable || relocation.relative || relocation.copy) {
                status = Relocation.Status.UNSUPPORTED;
            } else if (relocation.extern && (long)relocation.symbolNum < this.symtab.size()) {
                SymbolIterator symbolIterator = symbolTable.getSymbols(this.symtab.get((int)relocation.symbolNum).name);
                if (symbolIterator.hasNext()) {
                    value = symbolIterator.next().getAddress().getOffset();
                }
            } else if (!relocation.extern) {
                switch (relocation.symbolNum) {
                    case 4: {
                        value = textBlock.getStart().getOffset();
                        break;
                    }
                    case 6: {
                        value = dataBlock.getStart().getOffset();
                        break;
                    }
                    case 8: {
                        value = bssBlock.getStart().getOffset();
                    }
                }
            }
            if (value != null) {
                if (relocation.pcRelativeAddressing) {
                    value = value - targetBlock.getStart().getOffset();
                }
                byte[] newBytes = new byte[relocation.pointerLength];
                dc.putValue(value + addend, (int)relocation.pointerLength, newBytes, 0);
                targetBlock.putBytes(targetAddress, newBytes);
                status = Relocation.Status.APPLIED;
            }
            if (status != Relocation.Status.APPLIED) {
                this.log.appendMsg(targetBlock.getName(), String.format("Failed to apply relocation entry %d with type 0x%02x @ %s.", idx, relocation.flags, targetAddress));
            }
            relocationTable.add(targetAddress, status, (int)relocation.flags, new long[]{relocation.symbolNum}, originalBytes, relocation.getSymbolName(this.symtab));
            ++idx;
        }
    }

    private void markupSections() throws InvalidInputException, CodeUnitInsertionException, DuplicateNameException, IOException {
        MemoryBlock relDataBlock;
        AddressSpace defaultAddressSpace = this.program.getAddressFactory().getDefaultAddressSpace();
        FunctionManager functionManager = this.program.getFunctionManager();
        SymbolTable symbolTable = this.program.getSymbolTable();
        this.monitor.setMessage("Marking up header...");
        Address headerAddress = null;
        MemoryBlock aoutHeader = this.program.getMemory().getBlock("_aoutHeader");
        MemoryBlock textBlock = this.program.getMemory().getBlock(dot_text);
        if (aoutHeader != null) {
            headerAddress = aoutHeader.getStart();
        } else if (textBlock != null && this.header.getTextOffset() == 0L && this.header.getTextSize() >= 32L) {
            headerAddress = textBlock.getStart();
        }
        if (headerAddress != null) {
            this.header.markup(this.program, headerAddress);
        }
        if (this.header.getEntryPoint() != 0L) {
            Address address = defaultAddressSpace.getAddress(this.header.getEntryPoint());
            try {
                functionManager.createFunction("entry", address, (AddressSetView)new AddressSet(address), SourceType.IMPORTED);
            }
            catch (OverlappingFunctionException e) {
                this.log.appendMsg(dot_text, "Failed to create entrypoint function @ %s, creating symbol instead.");
                symbolTable.createLabel(address, "entry", SourceType.IMPORTED);
            }
        }
        this.monitor.setMessage("Marking up relocation tables...");
        MemoryBlock relTextBlock = this.program.getMemory().getBlock(dot_rel_text);
        if (relTextBlock != null) {
            this.relText.markup(this.program, relTextBlock);
        }
        if ((relDataBlock = this.program.getMemory().getBlock(dot_rel_data)) != null) {
            this.relData.markup(this.program, relDataBlock);
        }
        this.monitor.setMessage("Marking up symbol table...");
        MemoryBlock symtabBlock = this.program.getMemory().getBlock(dot_symtab);
        if (symtabBlock != null) {
            this.symtab.markup(this.program, symtabBlock);
        }
        this.monitor.setMessage("Marking up string table...");
        MemoryBlock strtabBlock = this.program.getMemory().getBlock(dot_strtab);
        if (strtabBlock != null) {
            this.strtab.markup(this.program, strtabBlock);
        }
    }
}

