# (c) Copyright 2009-2010, 2014-2015. CodeWeavers, Inc.

import re
import traceback

import cxlog
import cxutils

from gi.repository import GLib
from gi.repository import Gio


def not_much(*_args):
    pass


class Volume:
    device = ''
    # A string that uniquely identifies this device and does not change during
    # its lifetime. If possible, this is the filename of a block device.

    mounted = False
    # True if the device is currently mounted. This can lag behind the actual
    # device state, in which case calling mount, remount, or unmount will update
    # it.

    mountpoint = ''
    # If the device is mounted, this is its current mount point. This can change
    # if the device is mounted multiple times.

    label = ''
    # If possible, this is the label of the drive. An empty string otherwise.

    is_disc = False
    # True if the device is a CD or DVD.


_disc_filesystems = set(('cd9660',   # FreeBSD
                         'iso9660',  # Linux
                         'hsfs',     # Solaris
                         'udf',      # Linux
                         ))


class UDisks2Volume(Volume):
    is_published = False # True if we reported this with add_notify

    modified = False # True if the object was changed since add_notify was last called

    def __init__(self):
        Volume.__init__(self)

        self.connected_interfaces = {}


class UDisks2MountPoints:

    def __init__(self, add_notify=not_much, del_notify=not_much):
        self.object_manager = Gio.DBusObjectManagerClient.new_for_bus_sync(
            bus_type=Gio.BusType.SYSTEM,
            flags=Gio.DBusObjectManagerClientFlags.NONE,
            name='org.freedesktop.UDisks2',
            object_path='/org/freedesktop/UDisks2',
            get_proxy_type_func=None,
            get_proxy_type_user_data=None,
            cancellable=None)

        self.devices = {}

        for obj in self.object_manager.get_objects():
            self.object_changed(obj)

        self.object_manager.connect('interface-added', self.interface_added)
        self.object_manager.connect('interface-removed', self.interface_removed)

        self.add_notify = add_notify
        self.del_notify = del_notify

    def get_volume(self, objpath):
        try:
            volume = self.devices[objpath]
        except KeyError:
            volume = UDisks2Volume()
            self.devices[objpath] = volume

        return volume

    def object_changed(self, obj):
        obj_path = obj.get_object_path()
        cxlog.log('UDisks2 object changed at ' + obj_path)

        volume = self.get_volume(obj_path)

        interface_name = 'org.freedesktop.UDisks2.Block'
        block = obj.get_interface(interface_name)
        if block:
            if interface_name not in volume.connected_interfaces:
                volume.connected_interfaces[interface_name] = block.connect('g-properties-changed', self.properties_changed)

            device = block.get_cached_property('Device')
            if device:
                device = device.get_bytestring().decode('utf-8')

            if volume.is_published and volume.device != device:
                self.del_notify(volume)
                volume.is_published = False

            volume.device = device

            label = block.get_cached_property('IdLabel')
            if label:
                label = label.get_string()

            volume.modified = volume.modified or volume.label != label
            volume.label = label

            idtype = block.get_cached_property('IdType')
            if idtype:
                idtype = idtype.get_string()

            is_disc = idtype in _disc_filesystems
            volume.modified = volume.modified or volume.is_disc != is_disc
            volume.is_disc = is_disc
        elif interface_name in volume.connected_interfaces:
            del volume.connected_interfaces[interface_name]

        interface_name = 'org.freedesktop.UDisks2.Filesystem'
        fs = obj.get_interface(interface_name)
        if fs:
            if interface_name not in volume.connected_interfaces:
                volume.connected_interfaces[interface_name] = fs.connect('g-properties-changed', self.properties_changed)

            mountpoints = fs.get_cached_property('MountPoints')
            if mountpoints:
                mounted = True
                mountpoint = mountpoints.get_bytestring_array()[0]
            else:
                mounted = False
                mountpoint = ''

            if volume.is_published and (volume.mounted != mounted or volume.mountpoint != mountpoint):
                self.del_notify(volume)
                volume.is_published = False

            volume.mounted = mounted
            volume.mountpoint = mountpoint
        elif interface_name in volume.connected_interfaces:
            del volume.connected_interfaces[interface_name]

        cxlog.log('%s %s %s %s' % (str(volume.device), str(volume.label), str(volume.is_disc), str(volume.mountpoint)))

    def properties_changed(self, proxy, _props, _invalidated_props):
        self.object_changed(proxy.get_object())
        self.report_changes()

    def interface_added(self, _obj_manager, obj, _interface):
        self.object_changed(obj)
        self.report_changes()

    def interface_removed(self, _obj_manager, obj, _interface):
        self.object_changed(obj)
        self.report_changes()

    @staticmethod
    def should_report_volume(volume):
        if not volume.device:
            return False
        return volume.mounted

    def report_changes(self):
        for volume in self.devices.values():
            if self.should_report_volume(volume) != volume.is_published:
                if not volume.is_published:
                    self.add_notify(volume)
                    volume.is_published = True
                    volume.modified = False
                else:
                    self.del_notify(volume)
                    volume.is_published = False
            elif volume.is_published and volume.modified:
                self.add_notify(volume)
                volume.modified = False

    def close(self):
        self.add_notify = not_much
        self.del_notify = not_much

    def __del__(self):
        self.close()


class UnixVolume(Volume):
    # A mount point obtained from the mount command or /etc/fstab.
    pass


class FailsafeMountPoints:

    # The Linux mtab format assumes device paths don't contain spaces.
    # Otherwise it is ambiguous.
    LINUX_MTAB_RE = re.compile(r'(?P<device>\S+)\s+(?P<mountpoint>.+)\s+(?P<fs>\w+)\s+\S+\s+\d+\s+\d+$')
    SOLARIS_MTAB_RE = re.compile(r'(?P<device>.+)\t(?P<mountpoint>.+)\t(?P<fs>\w+)\t')
    LINUX_MOUNT_RE = re.compile(r'(?P<device>.+) on (?P<mountpoint>.+) type (?P<fs>\w+) \(.*\)')
    FREEBSD_MOUNT_RE = re.compile(r'(?P<device>.+) on (?P<mountpoint>.+) \((?P<fs>\w+)(?:,.*)?\)')

    def check_mount_cmd(self):
        mountdata = None
        for filename in ('/etc/mtab', '/etc/mnttab'):
            try:
                with open(filename, 'r', encoding='utf8', errors='surrogateescape') as thisfile: # pylint: disable=W1514
                    mountdata = list(thisfile.readlines())
                regexps = (self.LINUX_MTAB_RE, self.SOLARIS_MTAB_RE)
                break
            except IOError as ioerror:
                cxlog.log('could not read %s: %s' % (filename, ioerror))

        if mountdata is None:
            retcode, out, err = cxutils.run(('mount', ), stdout=cxutils.GRAB,
                                            stderr=cxutils.GRAB)
            if retcode:
                cxlog.log("mount failed (%s):\n%s%s" % (retcode, out, err))
            mountdata = out.split('\n')
            regexps = (self.LINUX_MOUNT_RE, self.FREEBSD_MOUNT_RE)

        known_devices = list(self.devices.keys())

        # Given the call parameters, communicate() always returns a string
        for line in mountdata:
            if line == '':
                continue

            for regexp in regexps:
                match = regexp.match(line)
                if match:
                    break
            if match is None:
                continue

            device = match.group('device')
            mp_path = match.group('mountpoint')
            filesystem = match.group('fs')

            if not device.startswith('/') or not mp_path.startswith('/') or \
                filesystem == 'sysfs':
                continue

            if device in known_devices:
                known_devices.remove(device)
            else:
                mountpoint = UnixVolume()
                mountpoint.device = device
                # Strange characters, spaces in particular, cause trouble in
                # mtab files and thus are escaped.
                mountpoint.mountpoint = cxutils.expand_octal_chars(mp_path)
                mountpoint.is_disc = filesystem in _disc_filesystems
                self.devices[device] = mountpoint
                self.add_notify(mountpoint)

        for device in known_devices:
            self.del_notify(self.devices[device])
            del self.devices[device]

        return True # continue timer

    def __init__(self, add_notify=not_much, del_notify=not_much):
        self.devices = {}

        self.add_notify = add_notify
        self.del_notify = del_notify

        self.timer_src = []

        self.check_mount_cmd()

        self.timer_src = [GLib.timeout_add(2000, self.check_mount_cmd)]

    def close(self):
        try:
            # using lists for cheap atomic operations, since this can run on multiple threads
            GLib.source_remove(self.timer_src.pop())
        except IndexError:
            pass

    def __del__(self):
        self.close()


# A list of notifier classes, with "better" notifiers at the start
notifier_classes = [
    UDisks2MountPoints,
    FailsafeMountPoints,
]


class MountPointsNotifier:
    "A class that returns the combined results from the full set of notifiers"
    def __init__(self, add_notify=not_much, del_notify=not_much):
        self.add_notify = add_notify
        self.del_notify = del_notify

        self.notifiers = []
        self.volumes = [{}]
        for notifier_class in notifier_classes:
            try:
                def on_add(volume, index=len(self.notifiers)):
                    self.on_add(volume, index)
                def on_del(volume, index=len(self.notifiers)):
                    self.on_del(volume, index)
                notifier = notifier_class(on_add, on_del)
                self.notifiers.append(notifier)
                self.volumes.append({})
                cxlog.log("using %s" % notifier_class)
            except Exception:
                cxlog.log("unable to create the %s object:\n%s" % (notifier_class.__name__, traceback.format_exc()))

        if not self.notifiers:
            cxlog.warn('could not create any mount point notifier')

        self.volumes.pop()

    def on_add(self, volume, index):
        self.volumes[index][volume.device] = volume

        for i in range(len(self.notifiers)):
            if volume.device in self.volumes[i]:
                if i < index:
                    # We already have a volume for this device, and it's better
                    return
                if i == index:
                    continue
                # This new device is better than the current best
                self.del_notify(self.volumes[i][volume.device])
                break

        self.add_notify(volume)

    def on_del(self, volume, index):
        del self.volumes[index][volume.device]
        self.del_notify(volume)

        for volumes in self.volumes:
            if volume.device in volumes:
                self.add_notify(volumes[volume.device])
                break

    def close(self):
        self.add_notify = not_much
        self.del_notify = not_much
        for notifier in self.notifiers:
            notifier.close()

    def _get_devices(self):
        devices_dict = {}
        for volumes in self.volumes[::-1]:
            devices_dict.update(volumes)
        return devices_dict

    devices = property(_get_devices)
