﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ecs/ECS_EXPORTS.h>
#include <aws/ecs/model/PlacementStrategyType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ECS {
namespace Model {

/**
 * <p>The task placement strategy for a task or service. For more information, see
 * <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html">Task
 * placement strategies</a> in the <i>Amazon Elastic Container Service Developer
 * Guide</i>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/PlacementStrategy">AWS
 * API Reference</a></p>
 */
class PlacementStrategy {
 public:
  AWS_ECS_API PlacementStrategy() = default;
  AWS_ECS_API PlacementStrategy(Aws::Utils::Json::JsonView jsonValue);
  AWS_ECS_API PlacementStrategy& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_ECS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The type of placement strategy. The <code>random</code> placement strategy
   * randomly places tasks on available candidates. The <code>spread</code> placement
   * strategy spreads placement across available candidates evenly based on the
   * <code>field</code> parameter. The <code>binpack</code> strategy places tasks on
   * available candidates that have the least available amount of the resource that's
   * specified with the <code>field</code> parameter. For example, if you binpack on
   * memory, a task is placed on the instance with the least amount of remaining
   * memory but still enough to run the task.</p>
   */
  inline PlacementStrategyType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(PlacementStrategyType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline PlacementStrategy& WithType(PlacementStrategyType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The field to apply the placement strategy against. For the
   * <code>spread</code> placement strategy, valid values are <code>instanceId</code>
   * (or <code>host</code>, which has the same effect), or any platform or custom
   * attribute that's applied to a container instance, such as
   * <code>attribute:ecs.availability-zone</code>. For the <code>binpack</code>
   * placement strategy, valid values are <code>cpu</code> and <code>memory</code>.
   * For the <code>random</code> placement strategy, this field is not used.</p>
   */
  inline const Aws::String& GetField() const { return m_field; }
  inline bool FieldHasBeenSet() const { return m_fieldHasBeenSet; }
  template <typename FieldT = Aws::String>
  void SetField(FieldT&& value) {
    m_fieldHasBeenSet = true;
    m_field = std::forward<FieldT>(value);
  }
  template <typename FieldT = Aws::String>
  PlacementStrategy& WithField(FieldT&& value) {
    SetField(std::forward<FieldT>(value));
    return *this;
  }
  ///@}
 private:
  PlacementStrategyType m_type{PlacementStrategyType::NOT_SET};
  bool m_typeHasBeenSet = false;

  Aws::String m_field;
  bool m_fieldHasBeenSet = false;
};

}  // namespace Model
}  // namespace ECS
}  // namespace Aws
