﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mediapackagev2/Mediapackagev2Request.h>
#include <aws/mediapackagev2/Mediapackagev2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace mediapackagev2 {
namespace Model {

/**
 */
class ListChannelsRequest : public Mediapackagev2Request {
 public:
  AWS_MEDIAPACKAGEV2_API ListChannelsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListChannels"; }

  AWS_MEDIAPACKAGEV2_API Aws::String SerializePayload() const override;

  AWS_MEDIAPACKAGEV2_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name that describes the channel group. The name is the primary identifier
   * for the channel group, and must be unique for your account in the AWS
   * Region.</p>
   */
  inline const Aws::String& GetChannelGroupName() const { return m_channelGroupName; }
  inline bool ChannelGroupNameHasBeenSet() const { return m_channelGroupNameHasBeenSet; }
  template <typename ChannelGroupNameT = Aws::String>
  void SetChannelGroupName(ChannelGroupNameT&& value) {
    m_channelGroupNameHasBeenSet = true;
    m_channelGroupName = std::forward<ChannelGroupNameT>(value);
  }
  template <typename ChannelGroupNameT = Aws::String>
  ListChannelsRequest& WithChannelGroupName(ChannelGroupNameT&& value) {
    SetChannelGroupName(std::forward<ChannelGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return in the response.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListChannelsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token from the GET list request. Use the token to fetch the
   * next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListChannelsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_channelGroupName;
  bool m_channelGroupNameHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace mediapackagev2
}  // namespace Aws
