"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Client = exports.HomerunnerError = void 0;
/*
Copyright 2022 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
const cross_fetch_1 = __importDefault(require("cross-fetch"));
class HomerunnerError extends Error {
    constructor(statusCode, body) {
        super(`Encountered an error while processing a homerunner request: ${statusCode} ${body}`);
    }
}
exports.HomerunnerError = HomerunnerError;
/**
 * A client interface for Homerunner.
 * @see https://github.com/matrix-org/complement/tree/main/cmd/homerunner
 */
class Client {
    /**
     * @param baseUrl The URL for homerunner's API endpoint. This will default to:
     * - The `HOMERUNNER_URL` env var, if defined.
     * - `http://localhost:${HOMERUNNER_PORT}`, if the HOMERUNNER_PORT env is defined.
     * - `http://localhost:54321`
     * @param fetch Fetch function compatable with the [Fetch API](https://developer.mozilla.org/en-US/docs/Web/API/Fetch_API). This will
     * use either the native fetch provided by Node 18+, or a polyfill.
     */
    constructor(baseUrl = process.env.HOMERUNNER_URL ||
        `http://localhost:${process.env.HOMERUNNER_PORT ?? 54321}`, fetch = global.fetch || cross_fetch_1.default) {
        this.baseUrl = baseUrl;
        this.fetch = fetch;
        if (baseUrl.endsWith('/')) {
            this.baseUrl = baseUrl.slice(0, -1);
        }
    }
    /**
     * Deploy a blueprint.
     * @param nameOrOptions Either a blueprint name that has been previously defined, or a in-line blueprint.
     */
    async create(nameOrOptions) {
        const options = typeof nameOrOptions === 'string' ? { blueprint_name: nameOrOptions } : nameOrOptions;
        const req = this.fetch(`${this.baseUrl}/create`, {
            method: 'POST',
            body: JSON.stringify(options),
            headers: {
                'Content-Type': 'application/json',
            },
        });
        const result = await req;
        if (result.status !== 200) {
            throw new HomerunnerError(result.status, await result.text());
        }
        return result.json();
    }
    /**
     * Destroy a blueprint.
     * @param blueprintName The name of the blueprint to destroy.
     */
    async destroy(blueprintName) {
        const req = this.fetch(`${this.baseUrl}/destroy`, {
            method: 'POST',
            body: JSON.stringify({ blueprint_name: blueprintName }),
            headers: {
                'Content-Type': 'application/json',
            },
        });
        const result = await req;
        if (result.status !== 200) {
            throw new HomerunnerError(result.status, await result.text());
        }
    }
    /**
     * Check to see if the homerunner service is up and listening for requests.
     */
    async health() {
        const req = this.fetch(`${this.baseUrl}/health`, {
            method: 'GET'
        });
        const result = await req;
        if (result.status !== 200) {
            throw new Error('Homerunner did not respond with an OK response');
        }
    }
}
exports.Client = Client;
//# sourceMappingURL=client.js.map